%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This script needs the following scripts to run
%	matcut.m	--> trims a matrix
%	veccut.m	--> trims a vector
%	fembeam.m	--> FEA for beams
%   beamsplitter --> splits beam into N beams
%   plotbeamelem.m--> plot beam elements
%   plotgap.m --> plot gaps
% It also needs the following input files
%	node.dat	--> 	nodal data
%	elem.dat	-->	element data
%	forces.dat	-->	force data
%	disp.dat	-->	displacement boundary conditon data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% READ INPUT from files
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read in nodal and element connectivity data from
% files: nodes.dat and elem.dat
%*************************** make gap.dat from elem.dat
%*************************** beamsplitter to split element into Ediv parts
clear
close
clc
Nmax = 100; % maximum number of nodes
[node,elem,gap]=beamsplitter(10);
% Read in force data from the file forces.dat
load forces.dat
% Read in displacement boundary condition data from the file disp.dat
load dispbc.dat
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PRE-PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Identify the number of nodes, X and Y Coordinates of the nodes
NNODE = size(node,1);
nx = node(:,2);
ny = node(:,3);
%********************************************************************
% Identify the Number of air-gaps, co-ordinates and voltage
Ngap = size(gap,1); 
gx1 = gap(:,2); 
gx2 = gap(:,4); % diagonal point x co-ordinate
gy1 = gap(:,3);
gy2 = gap(:,5); % diagonal point y co-ordinate
V = gap(:,6);
%find axial and transverse width
widthx = []; % axial width
widthy = []; % transverse width
for ig = 1:Ngap
    theta = gap(ig,7);
    rot = [cos(theta) -sin(theta); sin(theta) cos(theta)];
    A = rot'*[gx2(ig);gy2(ig)];
    widthx = [widthx;abs(gx1(ig)-A(1))];
    widthy = [widthy;abs(gy1(ig)-A(2))];
end
%*******************************************************************
% Identify the number of elements and form an element connectivity array,
% the cross-section and Young's modulus arrays.
NELEM = size(elem,1);
ncon = elem(:,[2 3]);
beam_width = elem(:,4);
beam_depth = elem(:,5);
E = elem(:,6);
A = beam_width.*beam_depth;
Inertia = beam_width.*(beam_depth.^3)/12;
%
% Arrange force information into a force vector, F
F = zeros(3*NNODE,1);			% Initialization
U = zeros(3*NNODE,1); % displacement vector
Nforce = size(forces,1);
for i = 1:Nforce,
     for in = 1:NNODE
        if force(i,2) == node(in,1);
            fnode = in;
        end
    end
    dof = (fnode-1)*3 + forces(i,3);
    F(dof) = forces(i,4);
end
% Displacement boundary conditions
Nfix = size(dispbc,1);
j = 0;
for i = 1:Nfix,
    j = j + 1;
     for in = 1:NNODE
        if dispbc(i,2) == node(in,1);
           dnode = in; 
        end
    end
    dispID(j) = (dnode-1)*3+dispbc(i,3);
    dispVal(j) = dispbc(i,4);
end
[dispID sortIndex] = sort(dispID);
dispVal = dispVal(sortIndex);
%% Compute the lengths of the elements
eye = []; %left node name
jay = []; % right node name
for ie=1:NELEM,
    %% get the index of the nodes
    for i = 1:NNODE
        if ncon(ie,1) == node(i,1);
            eye = [eye i];
        end
        if ncon(ie,2) == node(i,1)
            jay = [jay i];
        end
    end
    L(ie) = sqrt ( (nx(jay(ie)) - nx(eye(ie)))^2 + (ny(jay(ie)) - ny(eye(ie)))^2 );
end
%Begin the iterations
 pullin = []; % name of the gaps that go to pull-in.
 br = 0; %break variable
for iter = 1:25
    F = F*0;
    if br>0
        break;
    end
    iter

% compute the force due to gap immediately above or below an element
%**************************  its only vertical scout but use transverse scouts in advanced code
%************************* calculate only verticle(transverse) force now, axial later 
for ie = 1:NELEM
    for ig = 1:Ngap
        %% get the rotation vector
        theta  = gap(ig,7);
        rot = [cos(theta) -sin(theta); sin(theta) cos(theta)];
        %% rotate U(eye), nx,ny and diagonal
        ux = U(3*(eye(ie)-1)+1);
        uy = U(3*(eye(ie)-1)+2);
        ur = rot'*[ux;uy];      %rotated displacement
        umid = rot'*[ux+U(3*(jay(ie)-1)+1); uy+U(3*(jay(ie)-1)+2)]*0.5;
        nr = rot'*[nx(eye(ie))+nx(jay(ie));ny(eye(ie))+ny(jay(ie))]*0.5; %rotated node
        % rotate the gap diagonal
        gr = rot'*[gx2(ig);gy2(ig)];
        gmid = [gx1(ig)+gr(1) gy1(ig)+gr(2)]*0.5; % gap mid point
        % calculate transverse force..
        % gap can be below or above, but x bounds are common, u not needed
        gcondx = gx1(ig)<=nr(1)+ur(1) && nr(1)+ur(1)<=gr(1);
        gcondyup = gy1(ig)<nr(2)+0.01 && nr(2)+0.01<gr(2);
        gcondydown = gy1(ig)<nr(2)-0.01 && nr(2)-0.01<gr(2);
        gcondy = gcondyup||gcondydown;
        Fy = 0;
       if  gcondx && gcondy  % calculate and add transverse force
           if gcondyup  
               % forces for gap above
           Fy = 1e6*8.85e-12*L(ie)*beam_depth(ie)*V(ig)^2/(4*(widthy(ig)-umid(2))^2);
           %***************************1e6-unit corrector for force
            %************* Fringe field effect for the end beams
           if (eye(ie)<Nmax || jay(ie)<Nmax)
           Fy = Fy*(1+0.65)*(1-umid(2)/widthy(ig)); 
           end
           end
           if gcondydown
               % force for gap down; note the negative sign
           Fy = -1e6*8.85e-12*L(ie)*beam_depth(ie)*V(ig)^2/(4*(widthy(ig)+umid(2))^2);
            %************* Fringe field effect for the end beams
            if (eye(ie)<Nmax || jay(ie)<Nmax)
                Fy = Fy*(1+0.65)*(1+umid(2)/widthy(ig)); 
            end
           end
       end
           %**********************************************
           % calculate misalignment/axial force
           % this force depends on distance of beam element from gap center
           % beam should be above/below gap in transverse direction
           % fringe field not considered..
           Fx = 0;
           if gcondy
               Fx = 1e6*8.85e-12*beam_depth(ie)*V(ig)^2/(4*(widthy(ig)-umid(2)));
               Fx = Fx*sign(gmid(1)-nr(1)-umid(1));
           end
           %******************************************************
           %***************************1e6-unit corrector for force
           Fr = rot*[Fx;Fy];
           %*************************** apply rotation correction  
           % add transverse force
           F(3*(eye(ie)-1)+2)=F(3*(eye(ie)-1)+2)+Fr(2); 
           F(3*(jay(ie)-1)+2)=F(3*(jay(ie)-1)+2)+Fr(2);
           % add axial forces
           F(3*(eye(ie)-1)+1)=F(3*(eye(ie)-1)+1)+Fr(1);
           F(3*(jay(ie)-1)+1)=F(3*(jay(ie)-1)+1)+Fr(1);
           
           %% gaps that go to pull-in
           if gcondyup && widthy(ig)<3*ur(2) || gcondydown && widthy(ig)<-3*ur(2);
               pullin = [pullin ig];
           end
           % break if gap = 0
           if gcondyup && widthy(ig)<ur(2) || gcondydown && widthy(ig)<-ur(2);
               br = ig;
           end
     end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLUTION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Call fembeam.m to solve for the following.
%	Deflections: U
%	Reaction forces at the constrained nodes: R
%	Internal forces in each truss member: Fint
%	Global stiffness matrix: Kglobal
%	Strain energy: SE
[U,R,Fint,Fint_local,Kglobal,SE] = fembeam(A,L,E,nx,ny,ncon,NELEM,NNODE,F, ...
    dispID,dispVal,Inertia,eye,jay);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% POST-PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Plotting
clf
% The "scale factor" below is used for scaling the displacements for the purpose
% of plotting. Change it as needed.
scale_factor = 1;
Us = scale_factor * U;
for ip=1:NELEM,
    pt1 = eye(ip); pt2 = jay(ip);
    uelem = [Us(3*(pt1-1)+1); Us(3*(pt1-1)+2); Us(3*pt1); Us(3*(pt2-1)+1); Us(3*(pt2-1)+2); Us(3*pt2)];
    plotbeamelem(nx(pt1),ny(pt1),nx(pt2),ny(pt2),uelem,20);
    hold on
end
xlabel('X');
ylabel('Y');
%axis('equal');  % make x/y axis equal, real aspect ratio..

%***********************************************************
%************** call plotgap to plot the gaps
plotgap(gap);
grid on
pullin_gap = unique(pullin); % remove redundant entries
clear pullin
