%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Analysis of H and C matrices for trusses
% H = Force equilibrium matrix
% C = transpose of H = Displacement compatibility matrix
% Written by: G.K. Ananthasuresh
% originally for ME 851 in IIT-Kanpur in 2013;
% revised for ME260@IISc in 2020.
% Modified by Ripusudan Agrawal in Jan. 2024 for ME254@IISc
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clf                                    % Clear graphics window
clear all                              % Clear all variables
clc                                    % Clear command window
hold off                               % No hold on the graphics window
%
% This script needs the following scripts to run
%       matcut.m        --> trims a matrix
%       veccut.m        --> trims a vector
% It also needs the following input files
%       node.dat        --> nodal data
%       elem.dat        --> element data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% READ INPUT from files
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Included global variables from file globals.m
%
% Read in nodal and element connectivity data from
% files: nodes.dat and elem.dat
load node.dat
load elem.dat
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PRE-PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Identify the number of nodes, X and Y Coordinates of the nodes
NNODE = size(node,1);
nx = node(:,2);
ny = node(:,3);
%
% Identify the number of elements and form an element connectivity array,
% the cross-section and Young's modulus arrays.
NELEM = size(elem,1);
ncon = elem(:,[2 3]);
A = elem(:,4);
E = elem(:,5);
%
% Compute the lengths of the elements
for ie=1:NELEM,
    eye = ncon(ie,1);
    jay = ncon(ie,2);
    L(ie) = sqrt ( (nx(jay) - nx(eye))^2 + (ny(jay) - ny(eye))^2 );
end
%
[H] = getHmatrix(nx, ny, ncon, NELEM, NNODE);
C = H';
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% POST-PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Note: The elements in tension, compression, and no stress are
%               plotted in red, blue, and black respectively.
%               Undeformed configuration is plotted in black.
%               The nodes are marked with small yellow circles in the
%               undeformed configuration.
%
figure(1)
clf
% Plot undeformed truss
for ip=1:NELEM,
    pt1 = ncon(ip,1); pt2 = ncon(ip,2);
    h1 = plot([nx(pt1) nx(pt2)], [ny(pt1) ny(pt2)],':k','LineWidth',2);
    hold on
    plot([nx(pt1) nx(pt2)], [ny(pt1) ny(pt2)],'ok');
end
xlabel('X');
ylabel('Y');
axis('equal');
% Plot modes of self stress
T = null(H);
SoSS = size(T,2);
figure(2)
clf
for k = 1:1:SoSS,
    subplot(2,round(SoSS/2),k);
    for ip=1:NELEM,
        pt1 = ncon(ip,1); pt2 = ncon(ip,2);
        if T(ip,k) < 0,
            h1 = plot([nx(pt1) nx(pt2)], [ny(pt1) ny(pt2)],'-r','LineWidth',2);
        else
            h1 = plot([nx(pt1) nx(pt2)], [ny(pt1) ny(pt2)],'-b','LineWidth',2);
        end
        hold on
        val = num2str(T(ip,k));
        h1 = text( (nx(pt1)+nx(pt2))/2, (ny(pt1)+ny(pt2))/2,val );
    end
    axis equal
    axis off
end

title('States of self stress');
% Plot nullspace of the C matrix
SF = 1; % Scale factor to show the nullspace modes

%% Ripusudan's edit, taking first element to be fixed for supressisng rigid-body modes
modC = C;
modC(:,[2*ncon(1,1)-1:2*ncon(1,1),2*ncon(1,2)-1:2*ncon(1,2)])=[];

% Modfied C to be used from here for rigid-body mode supression
% change below line to null(C) for getting rigid modes
modU = SF*null(modC);
DoF = size(modU,2);
U=zeros(2*NNODE,DoF);
%%% adding back fixed nodes in the modU matrix
j=1;
for i=1:1:2*NNODE
    if all([2*ncon(1,1)-1:2*ncon(1,1),2*ncon(1,2)-1:2*ncon(1,2)]-i)
        U(i,:)=modU(j,:);
        j=j+1;
    end
end

%% Ripusudan's edit ends

figure(3)
clf

for k = 1:1:DoF,
    %U(:,k) = U(:,k) / max(abs(U(:,k)));
    %Ripu edit: not required as rigid modes are surpressed
    % %     U(1:2:2*NNODE,k) = U(1:2:2*NNODE,k) - U(1,k);
    % %     U(2:2:2*NNODE,k) = U(2:2:2*NNODE,k) - U(2,k);
    subplot(2,round(DoF/2),k);
    for ip=1:NELEM,
        pt1 = ncon(ip,1); pt2 = ncon(ip,2);
        eye1 = 2*(pt1-1)+1; jay1 = eye1 + 1;
        eye2 = 2*(pt2-1)+1; jay2 = eye2 + 1;
        h1 = plot([nx(pt1)+U(eye1,k) nx(pt2)+U(eye2,k)], ...
            [ny(pt1)+U(jay1,k) ny(pt2)+U(jay2,k)],'-r','LineWidth',2);
        hold on
        plot([nx(pt1) nx(pt2)], [ny(pt1) ny(pt2)],'.k');
        h1 = plot([nx(pt1) nx(pt2)], ...
            [ny(pt1) ny(pt2)],'--k');
    end
    axis('equal');
    axis off
end
title('Degrees of freedom');










