% Nonlinear corotational beam code for analyzing beam-based compliant
% mechanisms undergoing large displacements
%
% G. K. Ananthasuresh, Indian Institute of Science, Bengaluru, Sep. 2016
% Nonlinear beam code "nlfeambeam.m"  written by P. Sivanagendra in 2006,
% a student in the M2D2 lab in IISc.
%
% Pre and post processing file for running the nonlinear beam code
%
clf					% Clear graphics window
clear all	        % Clear all variables
clc					% Clear command window
hold off            % No hold on the graphics window
%
% This script needs the following scripts to run
%	matcut.m	--> trims a matrix
%	veccut.m	--> trims a vector
%	nlfembeam.m	--> FEA for 2D beam (small displacement analysis)
% It also needs the following input files
%	node.dat	--> nodal data
%	elem.dat	-->	element data
%	forces.dat	-->	force data
%	disp.dat	-->	displacement boundary conditon data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% READ INPUT from files
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Read in nodal and element connectivity data from
% files: nodes.dat and elem.dat
load node.dat
load elem.dat
%
% Read in force data from the file forces.dat
load forces.dat
% Read in displacement boundary condition data from the file disp.dat
load dispbc.dat
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PRE-PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Identify the number of nodes, X and Y Coordinates of the nodes
NNODE = size(node,1);
unit = 1;
nx = unit*node(:,2);
ny = unit*node(:,3);
%
% Identify the number of elements and form an element connectivity array,
% the cross-section and Young's modulus arrays.
NELEM = size(elem,1);
ncon = elem(:,[2 3]);
beam_width = elem(:,4);
Be = beam_width;
beam_depth = elem(:,5);
He = beam_depth;
Ee = elem(:,6);
A = beam_width.*beam_depth;
Inertia = beam_width.*(beam_depth.^3)/12;
Ine = Inertia;
%
% Arrange force information into a force vector, F
F = zeros(3*NNODE,1);			% Initialization
Nforce = size(forces,1);
for i = 1:Nforce,
    dof = (forces(i,2)-1)*3 + forces(i,3);
    F(dof) = forces(i,4);
end
%
% Displacement boundary conditions
Nfix = size(dispbc,1);
j = 0;
for i = 1:Nfix,
    j = j + 1;
    dispID(j) = (dispbc(i,2)-1)*3+dispbc(i,3);
    dispVal(j) = dispbc(i,4);
end
[dispID sortIndex] = sort(dispID);
dispVal = dispVal(sortIndex);
%
% Compute the lengths of the elements
for ie=1:NELEM,
    eye = ncon(ie,1);
    jay = ncon(ie,2);
    Le(ie) = sqrt ( (nx(jay) - nx(eye))^2 + (ny(jay) - ny(eye))^2 );
end

%
%%%%%%%%%%%%%%%%%
% Analysis
%%%%%%%%%%%%%%%%%
%
[U,GTANS,epsilon,sigma,strain,SE] = nlfembeam(He,Be,Le,Ee,nx,ny,ncon,NELEM, ...
    NNODE, F, dispID,dispVal,Ine);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% POST-PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Plotting
%clf
% The "scale factor" below is used for scaling the displacements for the purpose
% of plotting. Change it as needed.
scale_factor = 1;
Us = scale_factor * U;
% for ip=1:NELEM,
%     pt1 = ncon(ip,1); pt2 = ncon(ip,2);
%     dx1 = Us(3*(pt1-1)+1);
%     dy1 = Us(3*(pt1-1)+2);
%     dx2 = Us(3*(pt2-1)+1);
%     dy2 = Us(3*(pt2-1)+2);
%     %
%     plot([nx(pt1) nx(pt2)], [ny(pt1) ny(pt2)], '--k');
%     hold on
%     plot([nx(pt1)+dx1 nx(pt2)+dx2], [ny(pt1)+dy1 ny(pt2)+dy2], '-r');
% end
% xlabel('X');
% ylabel('Y');
% axis('equal');
% figure(2)
%clf
for ip=1:NELEM,
    pt1 = ncon(ip,1); pt2 = ncon(ip,2);
    %
    uelem = [Us(3*(pt1-1)+1); Us(3*(pt1-1)+2); Us(3*pt1); Us(3*(pt2-1)+1); Us(3*(pt2-1)+2); Us(3*pt2)];
    plotbeamelem(nx(pt1),ny(pt1),nx(pt2),ny(pt2),uelem,20);
    hold on
end
xlabel('X');
ylabel('Y');
axis('equal');
grid
title('Geometric Non-Linear Beam Model')
% text(0.05,0.02,['Output Disp in Y=',num2str(100*U(14)),'cm']);
% text(0.05,0.01,['Input Force in X=',num2str(F(4)),'N']);
% plot([-0.04 0.13],[0.12 0.12],'--b','LineWidth',2)


