%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Finite Element MATLAB script for
% deflection and stress analysis of trusses
% Written by: G.K. Ananthasuresh
% originally for MEAM 310 class in Spring 1997.
% and modified in the subsequent years.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
 clf                                    % Clear graphics window
 clear all                              % Clear all variables
 clc                                    % Clear command window
 hold off                               % No hold on the graphics window
%
% This script needs the following scripts to run
%       matcut.m        --> trims a matrix
%       veccut.m        --> trims a vector
%       femtruss.m      --> FEA for trusses
% It also needs the following input files
%       node.dat        --> nodal data
%       elem.dat        --> element data
%       forces.dat      --> force data
%       disp.dat        --> displacement boundary conditon data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% READ INPUT from files
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Included global variables from file globals.m
%
% Read in nodal and element connectivity data from
% files: nodes.dat and elem.dat
load node.dat
load elem.dat
%
% Read in force data from the file forces.dat
load forces.dat
% Read in displacement boundary condition data from the file disp.dat
load dispbc.dat
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PRE-PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Identify the number of nodes, X and Y Coordinates of the nodes
NNODE = size(node,1);
nx = node(:,2);
ny = node(:,3);
%
% Identify the number of elements and form an element connectivity array,
% the cross-section and Young's modulus arrays.
NELEM = size(elem,1);
ncon = elem(:,[2 3]);
A = elem(:,4);
E = elem(:,5);
%
% Arrange force information into a force vector, F
F = zeros(2*NNODE,1);                   % Initialization
Nforce = size(forces,1);
for i = 1:Nforce,
   dof = (forces(i,2)-1)*2 + forces(i,3);
   F(dof) = forces(i,4);
end
%
% Displacement boundary conditions
Nfix = size(dispbc,1);
j = 0;
for i = 1:Nfix,
    j = j + 1; dispID(j) = (dispbc(i,2)-1)*2+dispbc(i,3);
end
dispID = sort(dispID);
%
% Compute the lengths of the elements
for ie=1:NELEM,
  eye = ncon(ie,1);
  jay = ncon(ie,2);
  L(ie) = sqrt ( (nx(jay) - nx(eye))^2 + (ny(jay) - ny(eye))^2 );
end
%
%
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLUTION
 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Call femtruss.m to solve for the following.
%       Deflections: U
%       Reaction forces at the constrained nodes: R
%       Internal forces in each truss member: Fint
%       Global stiffness matrix, Kglobal
%       Strain energy: SE
[U,R,Fint,Kglobal,SE] = femtruss(A,L,E,nx,ny,ncon,NELEM,NNODE,F,dispID);
%
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% POST-PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Plotting
% Scale factor for deflection for showing the deformation
scale_fac = 1;
Us = U*scale_fac;
clf
for ip=1:NELEM,
  pt1 = ncon(ip,1); pt2 = ncon(ip,2);
  dx1 = Us(2*(pt1-1)+1);
  dy1 = Us(2*(pt1-1)+2);
  dx2 = Us(2*(pt2-1)+1);
  dy2 = Us(2*(pt2-1)+2);
  TorC = Fint(ip,5);
%
% Plot undeformed only
%
% Note: The elements in tension, compression, and no stress are
%               plotted in red, cyan, and white respectively.
%               Undeformed configuration is in dotted yellow.
%               The nodes are marked with little yellow circles in the
%               undeformed configuration.
%
  h1 = plot([nx(pt1) nx(pt2)], [ny(pt1) ny(pt2)],':k');
  hold on
  plot([nx(pt1) nx(pt2)], [ny(pt1) ny(pt2)],'ok');
  if TorC == 1
         h2 = plot([nx(pt1)+dx1 nx(pt2)+dx2], [ny(pt1)+dy1 
ny(pt2)+dy2],'-r');
  elseif TorC == -1
         h3 = plot([nx(pt1)+dx1 nx(pt2)+dx2], [ny(pt1)+dy1 
ny(pt2)+dy2],'-b');
  else
        h4 = plot([nx(pt1)+dx1 nx(pt2)+dx2], [ny(pt1)+dy1 ny(pt2)+dy2],'-w');
  end
end
xlabel('X');
ylabel('Y');
axis('equal');
legend([h1 h2 h3], 'Undeformed','Tensile stress',...
     'Compressive stress');

Strain_energy = SE
total_truss_length = sum(L)
metric = SE*sum(L)/20
